/**
 * Utility functions for Factory Efficiency Management System
 */

// API base URL
const API_BASE = '/api';

/**
 * Make an API request
 * @param {string} endpoint - API endpoint
 * @param {object} options - Fetch options
 * @returns {Promise} Fetch promise
 */
async function apiRequest(endpoint, options = {}) {
    const url = `${API_BASE}${endpoint}`;
    const defaultOptions = {
        headers: {
            'Content-Type': 'application/json',
        },
    };

    const finalOptions = { ...defaultOptions, ...options };

    try {
        const response = await fetch(url, finalOptions);
        const data = await response.json();
        return data;
    } catch (error) {
        console.error('API request failed:', error);
        throw error;
    }
}

/**
 * Show a toast notification
 * @param {string} message - Message to display
 * @param {string} type - Type of notification (success, danger, warning, info)
 */
function showToast(message, type = 'info') {
    // Create toast container if it doesn't exist
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        document.body.appendChild(container);
    }

    // Create toast element
    const toastId = `toast-${Date.now()}`;
    const toastHtml = `
        <div id="${toastId}" class="toast" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="toast-header bg-${type} text-white">
                <strong class="me-auto">${type === 'success' ? '成功' : type === 'danger' ? '错误' : '提示'}</strong>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                ${message}
            </div>
        </div>
    `;

    container.insertAdjacentHTML('beforeend', toastHtml);

    // Show the toast
    const toastEl = document.getElementById(toastId);
    const toast = new bootstrap.Toast(toastEl);
    toast.show();

    // Remove from DOM after hiding
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}

/**
 * Format a number with specified decimal places
 * @param {number} num - Number to format
 * @param {number} decimals - Number of decimal places
 * @returns {string} Formatted number
 */
function formatNumber(num, decimals = 2) {
    if (isNaN(num)) return '-';
    return num.toFixed(decimals);
}

/**
 * Format a date string
 * @param {string} dateString - Date string to format
 * @param {string} format - Format type ('short', 'long', 'time')
 * @returns {string} Formatted date
 */
function formatDate(dateString, format = 'short') {
    const date = new Date(dateString);

    if (isNaN(date.getTime())) return '-';

    switch (format) {
        case 'short':
            return date.toLocaleDateString('zh-CN');
        case 'long':
            return date.toLocaleDateString('zh-CN', {
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            });
        case 'time':
            return date.toLocaleString('zh-CN');
        default:
            return date.toLocaleDateString('zh-CN');
    }
}

/**
 * Confirm an action with a message
 * @param {string} message - Confirmation message
 * @returns {boolean} User's choice
 */
function confirmAction(message) {
    return confirm(message);
}

/**
 * Debounce a function
 * @param {function} func - Function to debounce
 * @param {number} wait - Wait time in milliseconds
 * @returns {function} Debounced function
 */
function debounce(func, wait = 300) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

/**
 * Get today's date in YYYY-MM-DD format
 * @returns {string} Today's date
 */
function getTodayDate() {
    return new Date().toISOString().split('T')[0];
}

/**
 * Get a date X days ago in YYYY-MM-DD format
 * @param {number} days - Number of days ago
 * @returns {string} Date string
 */
function getDateDaysAgo(days) {
    const date = new Date();
    date.setDate(date.getDate() - days);
    return date.toISOString().split('T')[0];
}

/**
 * Validate a form
 * @param {HTMLFormElement} form - Form element to validate
 * @returns {boolean} Is form valid
 */
function validateForm(form) {
    if (!form.checkValidity()) {
        form.reportValidity();
        return false;
    }
    return true;
}

/**
 * Disable a button and show loading state
 * @param {HTMLButtonElement} button - Button to disable
 * @param {string} originalText - Original button text
 */
function setButtonLoading(button, originalText) {
    button.disabled = true;
    button.dataset.originalText = originalText;
    button.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>处理中...';
}

/**
 * Re-enable a button after loading
 * @param {HTMLButtonElement} button - Button to re-enable
 */
function unsetButtonLoading(button) {
    button.disabled = false;
    const originalText = button.dataset.originalText || '提交';
    button.innerHTML = originalText;
}

/**
 * Export data to CSV
 * @param {array} data - Array of objects to export
 * @param {string} filename - Filename for download
 */
function exportToCSV(data, filename) {
    if (!data || data.length === 0) {
        showToast('没有数据可导出', 'warning');
        return;
    }

    // Get headers from first object
    const headers = Object.keys(data[0]);

    // Create CSV content
    const csvContent = [
        headers.join(','),
        ...data.map(row => headers.map(header => {
            const value = row[header];
            // Escape quotes and wrap in quotes if contains comma
            const stringValue = String(value || '');
            if (stringValue.includes(',') || stringValue.includes('"')) {
                return `"${stringValue.replace(/"/g, '""')}"`;
            }
            return stringValue;
        }).join(','))
    ].join('\n');

    // Create blob and download
    const blob = new Blob(['\ufeff' + csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Global error handler
window.addEventListener('error', (event) => {
    console.error('Global error:', event.error);
});

// Export for use in other files
window.EfficiencyUtils = {
    apiRequest,
    showToast,
    formatNumber,
    formatDate,
    confirmAction,
    debounce,
    getTodayDate,
    getDateDaysAgo,
    validateForm,
    setButtonLoading,
    unsetButtonLoading,
    exportToCSV
};
