"""
Dashboard routes for visualization and reporting
"""
from flask import Blueprint, render_template, request, jsonify, send_file
from flask_login import login_required, current_user
from database.models import db, DailyReport, DailyOutputDetail, DailyAbnormalDetail, ConfigDept
from datetime import datetime, date, timedelta
from sqlalchemy import func, and_
import pandas as pd
import io
import os

dashboard_bp = Blueprint('dashboard', __name__)


@dashboard_bp.route('/')
@login_required
def overview():
    """Dashboard overview page"""
    return render_template('dashboard/overview.html')


@dashboard_bp.route('/query')
@login_required
def query():
    """Report query page"""
    return render_template('dashboard/query.html')


@dashboard_bp.route('/api/efficiency-trend', methods=['GET'])
@login_required
def efficiency_trend():
    """Get efficiency trend data for charts"""
    # Get query parameters
    days = request.args.get('days', 30, type=int)
    department = request.args.get('department')
    line = request.args.get('line')

    # Calculate date range
    end_date = date.today()
    start_date = end_date - timedelta(days=days)

    # Build query
    query = DailyReport.query.filter(
        DailyReport.report_date >= start_date,
        DailyReport.report_date <= end_date
    )

    # Apply filters
    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config:
            query = query.filter_by(department=config.department, line=config.line)
    elif department:
        query = query.filter_by(department=department)
        if line:
            query = query.filter_by(line=line)

    # Get reports
    reports = query.order_by(DailyReport.report_date).all()

    # Prepare data for chart
    labels = [r.report_date.strftime('%Y-%m-%d') for r in reports]
    efficiency_data = [r.overall_efficiency or 0 for r in reports]
    direct_efficiency_data = [r.direct_efficiency or 0 for r in reports]
    indirect_efficiency_data = [r.indirect_efficiency or 0 for r in reports]

    return jsonify({
        'success': True,
        'data': {
            'labels': labels,
            'efficiency': efficiency_data,
            'direct_efficiency': direct_efficiency_data,
            'indirect_efficiency': indirect_efficiency_data
        }
    })


@dashboard_bp.route('/api/abnormal-distribution', methods=['GET'])
@login_required
def abnormal_distribution():
    """Get abnormal hours distribution for pie chart"""
    # Get query parameters
    days = request.args.get('days', 30, type=int)
    department = request.args.get('department')
    line = request.args.get('line')

    # Calculate date range
    end_date = date.today()
    start_date = end_date - timedelta(days=days)

    # Build query for abnormal details
    query = db.session.query(
        DailyAbnormalDetail.abnormal_type,
        func.sum(DailyAbnormalDetail.lost_hours).label('total_hours')
    ).join(
        DailyReport,
        DailyAbnormalDetail.report_id == DailyReport.id
    ).filter(
        DailyReport.report_date >= start_date,
        DailyReport.report_date <= end_date
    )

    # Apply filters
    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config:
            query = query.filter(
                DailyReport.department == config.department,
                DailyReport.line == config.line
            )
    else:
        if department:
            query = query.filter(DailyReport.department == department)
            if line:
                query = query.filter(DailyReport.line == line)

    # Group by abnormal type
    results = query.group_by(DailyAbnormalDetail.abnormal_type).all()

    # Prepare data for chart
    labels = [r[0] for r in results]
    data = [float(r[1]) for r in results]

    return jsonify({
        'success': True,
        'data': {
            'labels': labels,
            'values': data
        }
    })


@dashboard_bp.route('/api/reports', methods=['GET'])
@login_required
def get_reports():
    """Get filtered reports with pagination"""
    # Get query parameters
    page = request.args.get('page', 1, type=int)
    per_page = request.args.get('per_page', 20, type=int)
    date_from = request.args.get('date_from')
    date_to = request.args.get('date_to')
    department = request.args.get('department')
    line = request.args.get('line')

    # Build query
    query = DailyReport.query

    # Apply filters
    if date_from:
        try:
            query = query.filter(DailyReport.report_date >= datetime.strptime(date_from, '%Y-%m-%d').date())
        except ValueError:
            pass

    if date_to:
        try:
            query = query.filter(DailyReport.report_date <= datetime.strptime(date_to, '%Y-%m-%d').date())
        except ValueError:
            pass

    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config:
            query = query.filter_by(department=config.department, line=config.line)
    else:
        if department:
            query = query.filter_by(department=department)
            if line:
                query = query.filter_by(line=line)

    # Order and paginate
    query = query.order_by(DailyReport.report_date.desc(), DailyReport.department, DailyReport.line)
    pagination = query.paginate(page=page, per_page=per_page, error_out=False)

    return jsonify({
        'success': True,
        'data': [r.to_dict() for r in pagination.items],
        'total': pagination.total,
        'page': page,
        'per_page': per_page,
        'pages': pagination.pages
    })


@dashboard_bp.route('/api/export-excel', methods=['GET'])
@login_required
def export_excel():
    """Export reports to Excel file"""
    # Get query parameters
    date_from = request.args.get('date_from')
    date_to = request.args.get('date_to')
    department = request.args.get('department')
    line = request.args.get('line')

    # Build query
    query = DailyReport.query

    # Apply filters
    if date_from:
        try:
            query = query.filter(DailyReport.report_date >= datetime.strptime(date_from, '%Y-%m-%d').date())
        except ValueError:
            pass

    if date_to:
        try:
            query = query.filter(DailyReport.report_date <= datetime.strptime(date_to, '%Y-%m-%d').date())
        except ValueError:
            pass

    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config:
            query = query.filter_by(department=config.department, line=config.line)
    else:
        if department:
            query = query.filter_by(department=department)
            if line:
                query = query.filter_by(line=line)

    # Get all reports
    reports = query.order_by(DailyReport.report_date.desc(), DailyReport.department, DailyReport.line).all()

    # Prepare main report data for export
    data = []
    for r in reports:
        data.append({
            '日期': r.report_date.strftime('%Y-%m-%d'),
            '部门': r.department,
            '线别': r.line,
            '应出勤直接人力': r.expected_direct_manpower,
            '应出勤间接人力': r.expected_indirect_manpower,
            '实际出勤直接人力': r.actual_direct_attendance,
            '实际出勤间接人力': r.actual_indirect_attendance,
            '直接加班时数': r.direct_overtime_hours,
            '间接加班时数': r.indirect_overtime_hours,
            '直接调整时数': r.direct_adjustment_hours,
            '间接调整时数': r.indirect_adjustment_hours,
            '直接投入工时': r.direct_input_hours,
            '间接投入工时': r.indirect_input_hours,
            '总投入工时': r.total_input_hours,
            '直接产出工时': r.direct_output_hours,
            '间接产出工时': r.indirect_output_hours,
            '总产出工时': r.total_output_hours,
            '直接异常工时': r.direct_abnormal_hours,
            '间接异常工时': r.indirect_abnormal_hours,
            '总异常工时': r.total_abnormal_hours,
            '直接效率%': r.direct_efficiency,
            '间接效率%': r.indirect_efficiency,
            '整体效率%': r.overall_efficiency,
            '异常工时率%': r.abnormal_hours_rate,
            '创建人': r.created_by,
            '创建时间': r.created_at.strftime('%Y-%m-%d %H:%M:%S') if r.created_at else ''
        })

    # Prepare abnormal details data for export
    abnormal_data = []
    for r in reports:
        for ad in r.abnormal_details:
            abnormal_data.append({
                '日期': r.report_date.strftime('%Y-%m-%d'),
                '部门': r.department,
                '线别': r.line,
                '异常类型': ad.abnormal_type,
                '异常描述': ad.description or '',
                '直接异常工时': ad.direct_hours,
                '间接异常工时': ad.indirect_hours,
                '总异常工时': ad.lost_hours
            })

    # Create DataFrames and export to Excel
    df_report = pd.DataFrame(data)
    df_abnormal = pd.DataFrame(abnormal_data)

    # Create a bytes buffer for the Excel file
    output = io.BytesIO()
    with pd.ExcelWriter(output, engine='openpyxl') as writer:
        df_report.to_excel(writer, index=False, sheet_name='工时报表')
        if not df_abnormal.empty:
            df_abnormal.to_excel(writer, index=False, sheet_name='异常明细')

    output.seek(0)

    # Generate filename
    filename = f'efficiency_report_{date.today().strftime("%Y%m%d")}.xlsx'

    return send_file(
        output,
        mimetype='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        as_attachment=True,
        download_name=filename
    )
