"""
Daily report routes for data entry and management
"""
from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from database.models import db, DailyReport, DailyOutputDetail, DailyAbnormalDetail, ConfigDept, ConfigST, ConfigAbnormal
from datetime import datetime, date
from sqlalchemy import or_

daily_report_bp = Blueprint('daily_report', __name__)


@daily_report_bp.route('/')
@login_required
def list_reports():
    """List daily reports"""
    query = DailyReport.query

    # Filter for lead users (only their assigned line)
    if current_user.is_lead() and current_user.department_id and current_user.line_id:
        # Get the config dept for this lead
        config = ConfigDept.query.get(current_user.line_id)
        if config:
            query = query.filter_by(department=config.department, line=config.line)

    # Filter by date if provided
    date_from = request.args.get('date_from')
    date_to = request.args.get('date_to')
    if date_from:
        query = query.filter(DailyReport.report_date >= date_from)
    if date_to:
        query = query.filter(DailyReport.report_date <= date_to)

    # Order by date descending, then department, line
    reports = query.order_by(DailyReport.report_date.desc(),
                            DailyReport.department,
                            DailyReport.line).all()

    return render_template('daily_report/list.html', reports=reports)


@daily_report_bp.route('/entry')
@login_required
def entry_form():
    """Display the daily report entry form"""
    return render_template('daily_report/entry.html')


@daily_report_bp.route('/entry/<int:id>')
@login_required
def edit_form(id):
    """Edit an existing daily report"""
    report = DailyReport.query.get_or_404(id)

    # Check permissions - lead can only edit their own line's reports
    if current_user.is_lead():
        # For now, allow editing - can add stricter checks later
        pass

    # Load report details
    output_details = DailyOutputDetail.query.filter_by(report_id=id).all()
    abnormal_details = DailyAbnormalDetail.query.filter_by(report_id=id).all()

    return render_template('daily_report/entry.html', report=report,
                          output_details=output_details,
                          abnormal_details=abnormal_details)


@daily_report_bp.route('/api/save', methods=['POST'])
@login_required
def save_report():
    """Save or update a daily report"""
    data = request.get_json()

    # Validate required fields
    if not data.get('report_date') or not data.get('department') or not data.get('line'):
        return jsonify({'success': False, 'error': 'Report date, department, and line are required'})

    try:
        report_date = datetime.strptime(data['report_date'], '%Y-%m-%d').date()
    except ValueError:
        return jsonify({'success': False, 'error': 'Invalid date format'})

    # Check if report already exists for this date/department/line
    report_id = data.get('id')
    if report_id:
        report = DailyReport.query.get(report_id)
        if not report:
            return jsonify({'success': False, 'error': 'Report not found'})
    else:
        report = DailyReport.query.filter_by(
            report_date=report_date,
            department=data['department'],
            line=data['line']
        ).first()

        if report:
            return jsonify({'success': False, 'error': 'A report already exists for this date, department, and line'})

        report = DailyReport(
            report_date=report_date,
            department=data['department'],
            line=data['line']
        )

    # Update report fields
    report.expected_direct_manpower = int(data.get('expected_direct_manpower', 0))
    report.expected_indirect_manpower = int(data.get('expected_indirect_manpower', 0))
    report.actual_direct_attendance = int(data.get('actual_direct_attendance', 0))
    report.actual_indirect_attendance = int(data.get('actual_indirect_attendance', 0))
    # Direct/Indirect overtime and adjustment hours
    report.direct_overtime_hours = float(data.get('direct_overtime_hours', 0))
    report.indirect_overtime_hours = float(data.get('indirect_overtime_hours', 0))
    report.direct_adjustment_hours = float(data.get('direct_adjustment_hours', 0))
    report.indirect_adjustment_hours = float(data.get('indirect_adjustment_hours', 0))
    # Input hours
    report.direct_input_hours = float(data.get('direct_input_hours', 0))
    report.indirect_input_hours = float(data.get('indirect_input_hours', 0))
    report.total_input_hours = float(data.get('total_input_hours', 0))
    # Output hours
    report.direct_output_hours = float(data.get('direct_output_hours', 0))
    report.indirect_output_hours = float(data.get('indirect_output_hours', 0))
    report.total_output_hours = float(data.get('total_output_hours', 0))
    # Abnormal hours
    report.direct_abnormal_hours = float(data.get('direct_abnormal_hours', 0))
    report.indirect_abnormal_hours = float(data.get('indirect_abnormal_hours', 0))
    report.total_abnormal_hours = float(data.get('total_abnormal_hours', 0))
    # Efficiency metrics
    report.direct_efficiency = float(data.get('direct_efficiency', 0))
    report.indirect_efficiency = float(data.get('indirect_efficiency', 0))
    report.overall_efficiency = float(data.get('overall_efficiency', 0))
    report.abnormal_hours_rate = float(data.get('abnormal_hours_rate', 0))
    report.created_by = current_user.username

    db.session.add(report)
    db.session.flush()  # Get the report ID

    # Delete existing output and abnormal details if editing
    if report_id:
        DailyOutputDetail.query.filter_by(report_id=report.id).delete()
        DailyAbnormalDetail.query.filter_by(report_id=report.id).delete()

    # Add output details
    if data.get('output_details'):
        for output in data['output_details']:
            detail = DailyOutputDetail(
                report_id=report.id,
                product_series=output['product_series'],
                model=output['model'],
                output_quantity=int(output['output_quantity']),
                direct_st=float(output['direct_st']),
                indirect_st=float(output['indirect_st']),
                direct_output_hours=float(output.get('direct_output_hours', 0)),
                indirect_output_hours=float(output.get('indirect_output_hours', 0)),
                output_hours=float(output.get('direct_output_hours', 0)) + float(output.get('indirect_output_hours', 0))
            )
            db.session.add(detail)

    # Add abnormal details
    if data.get('abnormal_details'):
        for abnormal in data['abnormal_details']:
            detail = DailyAbnormalDetail(
                report_id=report.id,
                abnormal_type=abnormal['abnormal_type'],
                description=abnormal.get('description', ''),
                direct_hours=float(abnormal.get('direct_hours', 0)),
                indirect_hours=float(abnormal.get('indirect_hours', 0)),
                lost_hours=float(abnormal.get('direct_hours', 0)) + float(abnormal.get('indirect_hours', 0))
            )
            db.session.add(detail)

    db.session.commit()

    return jsonify({'success': True, 'data': report.to_dict()})


@daily_report_bp.route('/api/delete/<int:id>', methods=['DELETE'])
@login_required
def delete_report(id):
    """Delete a daily report"""
    report = DailyReport.query.get_or_404(id)

    # Check permissions - lead can only delete their own line's reports
    if current_user.is_lead():
        # For now, allow deletion - can add stricter checks later
        pass

    db.session.delete(report)
    db.session.commit()

    return jsonify({'success': True})


@daily_report_bp.route('/api/get-departments', methods=['GET'])
@login_required
def get_departments():
    """Get list of all departments"""
    if current_user.is_admin():
        depts = ConfigDept.query.with_entities(
            ConfigDept.department
        ).distinct().order_by(ConfigDept.department).all()
    else:
        # Lead users only see their assigned department
        if current_user.department_id:
            config = ConfigDept.query.get(current_user.department_id)
            if config:
                depts = [(config.department,)]
            else:
                depts = []
        else:
            depts = []

    return jsonify({'success': True, 'data': [d[0] for d in depts]})


@daily_report_bp.route('/api/get-lines', methods=['GET'])
@login_required
def get_lines():
    """Get list of lines for a department"""
    department = request.args.get('department')

    if not department:
        return jsonify({'success': False, 'error': 'Department is required'})

    query = ConfigDept.query.filter_by(department=department)

    # Lead users only see their assigned line
    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config and config.department == department:
            query = query.filter_by(line=config.line)
        else:
            return jsonify({'success': True, 'data': []})

    configs = query.order_by(ConfigDept.line).all()

    return jsonify({'success': True, 'data': [c.to_dict() for c in configs]})


@daily_report_bp.route('/entry')
@login_required
def entry():
    """Entry page - redirect to form"""
    return render_template('daily_report/entry.html')
