"""
Configuration routes for admin (Department, Standard Hours, Abnormal Types)
"""
from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for
from flask_login import login_required, current_user
from database.models import db, ConfigDept, ConfigST, ConfigAbnormal
from routes.auth import admin_required
from datetime import datetime

config_bp = Blueprint('config', __name__)


# ============================================================================
# Department Configuration
# ============================================================================

@config_bp.route('/department')
@login_required
@admin_required
def department():
    """List all department configurations"""
    depts = ConfigDept.query.order_by(ConfigDept.department, ConfigDept.line).all()
    return render_template('config/department.html', depts=depts)


@config_bp.route('/department/add', methods=['POST'])
@login_required
@admin_required
def department_add():
    """Add a new department configuration"""
    data = request.get_json()

    # Validate required fields
    required_fields = ['department', 'line', 'expected_direct_manpower',
                      'expected_indirect_manpower', 'normal_working_hours']
    for field in required_fields:
        if field not in data or not data[field]:
            return jsonify({'success': False, 'error': f'{field} is required'})

    # Check for uniqueness
    existing = ConfigDept.query.filter_by(
        department=data['department'],
        line=data['line']
    ).first()

    if existing:
        return jsonify({'success': False, 'error': 'This department and line combination already exists'})

    # Create new configuration
    config = ConfigDept(
        department=data['department'],
        line=data['line'],
        expected_direct_manpower=int(data['expected_direct_manpower']),
        expected_indirect_manpower=int(data['expected_indirect_manpower']),
        normal_working_hours=float(data['normal_working_hours'])
    )

    db.session.add(config)
    db.session.commit()

    return jsonify({'success': True, 'data': config.to_dict()})


@config_bp.route('/department/<int:id>', methods=['PUT', 'DELETE'])
@login_required
@admin_required
def department_detail(id):
    """Update or delete a department configuration"""
    config = ConfigDept.query.get_or_404(id)

    if request.method == 'DELETE':
        db.session.delete(config)
        db.session.commit()
        return jsonify({'success': True})

    # PUT request - update
    data = request.get_json()

    # Check uniqueness if department or line is changing
    if (data.get('department') != config.department or
        data.get('line') != config.line):
        existing = ConfigDept.query.filter_by(
            department=data['department'],
            line=data['line']
        ).first()
        if existing and existing.id != id:
            return jsonify({'success': False, 'error': 'This department and line combination already exists'})

    config.department = data.get('department', config.department)
    config.line = data.get('line', config.line)
    config.expected_direct_manpower = int(data.get('expected_direct_manpower', config.expected_direct_manpower))
    config.expected_indirect_manpower = int(data.get('expected_indirect_manpower', config.expected_indirect_manpower))
    config.normal_working_hours = float(data.get('normal_working_hours', config.normal_working_hours))
    config.updated_at = datetime.utcnow()

    db.session.commit()

    return jsonify({'success': True, 'data': config.to_dict()})


@config_bp.route('/department/validate-line', methods=['GET'])
@login_required
@admin_required
def validate_line():
    """Validate line uniqueness for a department"""
    department = request.args.get('department')
    line = request.args.get('line')
    exclude_id = request.args.get('exclude_id', type=int)

    if not department or not line:
        return jsonify({'valid': False})

    query = ConfigDept.query.filter_by(department=department, line=line)
    if exclude_id:
        query = query.filter(ConfigDept.id != exclude_id)

    existing = query.first()

    return jsonify({'valid': existing is None})


@config_bp.route('/api/all-lines', methods=['GET'])
@login_required
@admin_required
def get_all_lines():
    """Get all lines from department configuration for dropdown"""
    configs = ConfigDept.query.order_by(ConfigDept.department, ConfigDept.line).all()

    # Format as list of {line, department_line} for display
    lines = []
    for config in configs:
        lines.append({
            'line': config.line,
            'department': config.department,
            'display': f"{config.department} - {config.line}"
        })

    return jsonify({'success': True, 'data': lines})


# ============================================================================
# Standard Hours Configuration
# ============================================================================

@config_bp.route('/standard-hours')
@login_required
@admin_required
def standard_hours():
    """List all standard hours configurations"""
    st_configs = ConfigST.query.order_by(ConfigST.product_series, ConfigST.model).all()
    return render_template('config/standard_hours.html', st_configs=st_configs)


@config_bp.route('/standard-hours/add', methods=['POST'])
@login_required
@admin_required
def st_add():
    """Add a new standard hours configuration"""
    data = request.get_json()

    # Validate required fields
    required_fields = ['product_series', 'model', 'direct_st', 'indirect_st', 'applicable_line']
    for field in required_fields:
        if field not in data or data[field] is None or data[field] == '':
            return jsonify({'success': False, 'error': f'{field} is required'})

    # Create new configuration
    config = ConfigST(
        product_series=data['product_series'],
        model=data['model'],
        direct_st=float(data['direct_st']),
        indirect_st=float(data['indirect_st']),
        applicable_line=data['applicable_line']
    )

    db.session.add(config)
    db.session.commit()

    return jsonify({'success': True, 'data': config.to_dict()})


@config_bp.route('/standard-hours/<int:id>', methods=['PUT', 'DELETE'])
@login_required
@admin_required
def st_detail(id):
    """Update or delete a standard hours configuration"""
    config = ConfigST.query.get_or_404(id)

    if request.method == 'DELETE':
        db.session.delete(config)
        db.session.commit()
        return jsonify({'success': True})

    # PUT request - update
    data = request.get_json()

    config.product_series = data.get('product_series', config.product_series)
    config.model = data.get('model', config.model)
    config.direct_st = float(data.get('direct_st', config.direct_st))
    config.indirect_st = float(data.get('indirect_st', config.indirect_st))
    config.applicable_line = data.get('applicable_line', config.applicable_line)
    config.updated_at = datetime.utcnow()

    db.session.commit()

    return jsonify({'success': True, 'data': config.to_dict()})


# ============================================================================
# Abnormal Type Configuration
# ============================================================================

@config_bp.route('/abnormal')
@login_required
@admin_required
def abnormal():
    """List all abnormal type configurations"""
    abnormal_types = ConfigAbnormal.query.order_by(ConfigAbnormal.name).all()
    return render_template('config/abnormal.html', abnormal_types=abnormal_types)


@config_bp.route('/abnormal/add', methods=['POST'])
@login_required
@admin_required
def abnormal_add():
    """Add a new abnormal type"""
    data = request.get_json()

    if not data.get('name'):
        return jsonify({'success': False, 'error': 'Name is required'})

    # Check for uniqueness
    existing = ConfigAbnormal.query.filter_by(name=data['name']).first()
    if existing:
        return jsonify({'success': False, 'error': 'This abnormal type name already exists'})

    config = ConfigAbnormal(
        name=data['name'],
        description=data.get('description', ''),
        is_active=data.get('is_active', True)
    )

    db.session.add(config)
    db.session.commit()

    return jsonify({'success': True, 'data': config.to_dict()})


@config_bp.route('/abnormal/<int:id>', methods=['PUT', 'DELETE'])
@login_required
@admin_required
def abnormal_detail(id):
    """Update or delete an abnormal type"""
    config = ConfigAbnormal.query.get_or_404(id)

    if request.method == 'DELETE':
        # Check if this abnormal type is in use
        from database.models import DailyAbnormalDetail
        in_use = DailyAbnormalDetail.query.filter_by(abnormal_type=config.name).first()
        if in_use:
            return jsonify({'success': False, 'error': 'Cannot delete: This abnormal type is in use'})

        db.session.delete(config)
        db.session.commit()
        return jsonify({'success': True})

    # PUT request - update
    data = request.get_json()

    # Check uniqueness if name is changing
    if data.get('name') != config.name:
        existing = ConfigAbnormal.query.filter_by(name=data['name']).first()
        if existing:
            return jsonify({'success': False, 'error': 'This abnormal type name already exists'})

    config.name = data.get('name', config.name)
    config.description = data.get('description', config.description)
    config.is_active = data.get('is_active', config.is_active)

    db.session.commit()

    return jsonify({'success': True, 'data': config.to_dict()})
