"""
Authentication routes for login/logout
"""
from flask import Blueprint, render_template, request, redirect, url_for, flash, jsonify
from flask_login import login_user, logout_user, login_required, current_user
from database.models import db, User, ConfigDept

auth_bp = Blueprint('auth', __name__)


@auth_bp.route('/login', methods=['GET', 'POST'])
def login():
    """Handle user login"""
    if request.method == 'POST':
        username = request.form.get('username', '').strip()
        password = request.form.get('password', '')
        remember = request.form.get('remember', False)

        if not username or not password:
            flash('Please enter username and password.', 'danger')
            return render_template('login.html')

        # Find user by username
        user = User.query.filter_by(username=username).first()

        # Check password
        if user and user.check_password(password):
            login_user(user, remember=bool(remember))
            flash(f'Welcome, {user.username}!', 'success')

            # Redirect to the page they were trying to access
            next_page = request.args.get('next')
            if next_page:
                return redirect(next_page)
            return redirect(url_for('index'))
        else:
            flash('Invalid username or password.', 'danger')

    return render_template('login.html')


@auth_bp.route('/logout')
@login_required
def logout():
    """Handle user logout"""
    logout_user()
    flash('You have been logged out.', 'info')
    return redirect(url_for('auth.login'))


@auth_bp.route('/users')
@login_required
def list_users():
    """List all users (admin only)"""
    if not current_user.is_admin():
        flash('Access denied. Admin only.', 'danger')
        return redirect(url_for('index'))

    users = User.query.order_by(User.created_at.desc()).all()
    return render_template('admin/users.html', users=users)


@auth_bp.route('/api/users', methods=['POST'])
@login_required
def create_user():
    """Create a new user (admin only)"""
    if not current_user.is_admin():
        return jsonify({'success': False, 'error': 'Access denied'}), 403

    data = request.get_json()

    # Validate required fields
    if not data.get('username'):
        return jsonify({'success': False, 'error': 'Username is required'})
    if not data.get('password'):
        return jsonify({'success': False, 'error': 'Password is required'})
    if not data.get('role'):
        return jsonify({'success': False, 'error': 'Role is required'})

    # Check if username already exists
    existing = User.query.filter_by(username=data['username']).first()
    if existing:
        return jsonify({'success': False, 'error': 'Username already exists'})

    # For lead users, validate department and line
    if data.get('role') == 'lead':
        if not data.get('department_id'):
            return jsonify({'success': False, 'error': 'Department is required for lead users'})
        if not data.get('line_id'):
            return jsonify({'success': False, 'error': 'Line is required for lead users'})

        # Verify the department and line exist
        config = ConfigDept.query.get(data['department_id'])
        if not config:
            return jsonify({'success': False, 'error': 'Invalid department'})

    # Create new user
    user = User(
        username=data['username'],
        role=data['role'],
        department_id=data.get('department_id'),
        line_id=data.get('line_id')
    )
    user.set_password(data['password'])

    db.session.add(user)
    db.session.commit()

    return jsonify({'success': True, 'data': user.to_dict()})


@auth_bp.route('/api/users/<int:id>', methods=['PUT', 'DELETE'])
@login_required
def user_detail(id):
    """Update or delete a user (admin only)"""
    if not current_user.is_admin():
        return jsonify({'success': False, 'error': 'Access denied'}), 403

    user = User.query.get_or_404(id)

    if request.method == 'DELETE':
        # Prevent deleting yourself
        if user.id == current_user.id:
            return jsonify({'success': False, 'error': 'Cannot delete your own account'})
        db.session.delete(user)
        db.session.commit()
        return jsonify({'success': True})

    # PUT request - update
    data = request.get_json()

    # Check username uniqueness if changing
    if data.get('username') != user.username:
        existing = User.query.filter_by(username=data['username']).first()
        if existing:
            return jsonify({'success': False, 'error': 'Username already exists'})

    # Update fields
    user.username = data.get('username', user.username)
    user.role = data.get('role', user.role)

    # Update password if provided
    if data.get('password'):
        user.set_password(data['password'])

    # Update department and line for lead users
    if user.role == 'lead':
        user.department_id = data.get('department_id')
        user.line_id = data.get('line_id')
    else:
        # Clear department and line for admin users
        user.department_id = None
        user.line_id = None

    db.session.commit()

    return jsonify({'success': True, 'data': user.to_dict()})


def admin_required(f):
    """Decorator to require admin access"""
    from functools import wraps
    from flask_login import current_user
    from flask import abort

    @wraps(f)
    def decorated_function(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin():
            abort(403)
        return f(*args, **kwargs)
    return decorated_function
