"""
API routes for AJAX calls from frontend
"""
from flask import Blueprint, jsonify, request
from flask_login import login_required, current_user
from database.models import db, ConfigDept, ConfigST, ConfigAbnormal, User

api_bp = Blueprint('api', __name__)


# ============================================================================
# Configuration APIs for daily report entry
# ============================================================================

@api_bp.route('/config-for-line', methods=['GET'])
@login_required
def get_config_for_line():
    """Get configuration data for a specific department and line"""
    department = request.args.get('dept')
    line = request.args.get('line')

    if not department or not line:
        return jsonify({'success': False, 'error': 'Department and line are required'})

    config = ConfigDept.query.filter_by(department=department, line=line).first()

    if not config:
        return jsonify({'success': False, 'error': 'Configuration not found for this department and line'})

    return jsonify({
        'success': True,
        'data': config.to_dict()
    })


@api_bp.route('/models-for-line', methods=['GET'])
@login_required
def get_models_for_line():
    """Get available standard hours models for a specific line"""
    line = request.args.get('line')

    if not line:
        return jsonify({'success': False, 'error': 'Line is required'})

    print(f"[DEBUG] Fetching models for line: '{line}'")

    # Get all ST configs for this line
    st_configs = ConfigST.query.filter_by(applicable_line=line).order_by(
        ConfigST.product_series, ConfigST.model
    ).all()

    print(f"[DEBUG] Found {len(st_configs)} ST configs for line '{line}'")
    for st in st_configs:
        print(f"  - {st.product_series}: {st.model} (direct_st={st.direct_st}, indirect_st={st.indirect_st})")

    # Group by product series
    result = {}
    for st in st_configs:
        if st.product_series not in result:
            result[st.product_series] = []
        result[st.product_series].append(st.to_dict())

    print(f"[DEBUG] Returning grouped data: {list(result.keys())}")

    return jsonify({
        'success': True,
        'data': result
    })


@api_bp.route('/abnormal-types', methods=['GET'])
@login_required
def get_abnormal_types():
    """Get all active abnormal types"""
    abnormal_types = ConfigAbnormal.query.filter_by(is_active=True).order_by(
        ConfigAbnormal.name
    ).all()

    return jsonify({
        'success': True,
        'data': [at.to_dict() for at in abnormal_types]
    })


@api_bp.route('/departments', methods=['GET'])
@login_required
def get_departments():
    """Get list of all departments"""
    if current_user.is_admin():
        depts = ConfigDept.query.with_entities(
            ConfigDept.department
        ).distinct().order_by(ConfigDept.department).all()
    else:
        # Lead users only see their assigned department
        if current_user.department_id:
            config = ConfigDept.query.get(current_user.department_id)
            if config:
                depts = [(config.department,)]
            else:
                depts = []
        else:
            depts = []

    return jsonify({'success': True, 'data': [d[0] for d in depts]})


@api_bp.route('/lines', methods=['GET'])
@login_required
def get_lines():
    """Get list of lines for a department"""
    department = request.args.get('department')

    if not department:
        return jsonify({'success': False, 'error': 'Department is required'})

    query = ConfigDept.query.filter_by(department=department)

    # Lead users only see their assigned line
    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config and config.department == department:
            query = query.filter_by(line=config.line)
        else:
            return jsonify({'success': True, 'data': []})

    configs = query.order_by(ConfigDept.line).all()

    return jsonify({'success': True, 'data': [c.to_dict() for c in configs]})


# ============================================================================
# Statistics APIs
# ============================================================================

@api_bp.route('/stats/summary', methods=['GET'])
@login_required
def get_stats_summary():
    """Get summary statistics for dashboard"""
    from datetime import date, timedelta
    from database.models import DailyReport

    # Get today's date
    today = date.today()
    yesterday = today - timedelta(days=1)

    # Build base query
    query = DailyReport.query

    # Filter for lead users
    if current_user.is_lead() and current_user.line_id:
        config = ConfigDept.query.get(current_user.line_id)
        if config:
            query = query.filter_by(department=config.department, line=config.line)

    # Get today's reports
    today_reports = query.filter(DailyReport.report_date == today).all()

    # Get yesterday's reports
    yesterday_reports = query.filter(DailyReport.report_date == yesterday).all()

    # Calculate statistics
    today_count = len(today_reports)
    yesterday_count = len(yesterday_reports)

    # Average efficiency for today
    if today_reports:
        avg_efficiency = sum(r.overall_efficiency or 0 for r in today_reports) / len(today_reports)
    else:
        avg_efficiency = 0

    # Average abnormal rate for today
    if today_reports:
        avg_abnormal_rate = sum(r.abnormal_hours_rate or 0 for r in today_reports) / len(today_reports)
    else:
        avg_abnormal_rate = 0

    return jsonify({
        'success': True,
        'data': {
            'today_count': today_count,
            'yesterday_count': yesterday_count,
            'avg_efficiency': round(avg_efficiency, 2),
            'avg_abnormal_rate': round(avg_abnormal_rate, 2)
        }
    })


@api_bp.route('/current-user', methods=['GET'])
@login_required
def get_current_user():
    """Get current user information"""
    return jsonify({
        'success': True,
        'data': {
            'id': current_user.id,
            'username': current_user.username,
            'role': current_user.role,
            'is_admin': current_user.is_admin(),
            'is_lead': current_user.is_lead(),
            'department_id': current_user.department_id,
            'line_id': current_user.line_id
        }
    })
