"""
Database models for the Factory Efficiency Management System
"""
from flask_sqlalchemy import SQLAlchemy
from datetime import datetime
from werkzeug.security import generate_password_hash, check_password_hash

db = SQLAlchemy()


class User(db.Model):
    """User model for authentication"""
    __tablename__ = 'users'

    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(255), nullable=False)
    role = db.Column(db.String(20), nullable=False)  # 'admin' or 'lead'
    department_id = db.Column(db.Integer, nullable=True)  # For lead users
    line_id = db.Column(db.Integer, nullable=True)  # For lead users
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    def set_password(self, password):
        """Hash and set the user's password"""
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        """Verify the user's password"""
        return check_password_hash(self.password_hash, password)

    def is_admin(self):
        """Check if user is an admin"""
        return self.role == 'admin'

    def is_lead(self):
        """Check if user is a lead"""
        return self.role == 'lead'

    # Flask-Login required properties
    @property
    def is_active(self):
        """Flask-Login: User is always active"""
        return True

    @property
    def is_authenticated(self):
        """Flask-Login: User is authenticated"""
        return True

    def get_id(self):
        """Flask-Login: Return user ID as string"""
        return str(self.id)

    def get_id(self):
        """Flask-Login: Return user ID as string"""
        return str(self.id)

    def to_dict(self):
        """Convert to dictionary"""
        result = {
            'id': self.id,
            'username': self.username,
            'role': self.role,
            'is_admin': self.is_admin(),
            'is_lead': self.is_lead(),
            'department_id': self.department_id,
            'line_id': self.line_id,
            'created_at': self.created_at.strftime('%Y-%m-%d %H:%M:%S') if self.created_at else None,
        }
        # Add department and line info if available
        if self.department_id:
            from database.models import ConfigDept
            config = ConfigDept.query.get(self.department_id)
            if config:
                result['department'] = config.department
                result['line'] = config.line
        return result

    def __repr__(self):
        return f'<User {self.username} ({self.role})>'


class ConfigDept(db.Model):
    """Department manpower configuration"""
    __tablename__ = 'configs_dept'

    id = db.Column(db.Integer, primary_key=True)
    department = db.Column(db.String(100), nullable=False)
    line = db.Column(db.String(100), nullable=False)
    expected_direct_manpower = db.Column(db.Integer, nullable=False)
    expected_indirect_manpower = db.Column(db.Integer, nullable=False, default=0)
    normal_working_hours = db.Column(db.Float, nullable=False, default=8.0)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)

    # Unique constraint on department and line combination
    __table_args__ = (
        db.UniqueConstraint('department', 'line', name='uq_dept_line'),
    )

    def to_dict(self):
        """Convert to dictionary"""
        return {
            'id': self.id,
            'department': self.department,
            'line': self.line,
            'expected_direct_manpower': self.expected_direct_manpower,
            'expected_indirect_manpower': self.expected_indirect_manpower,
            'normal_working_hours': self.normal_working_hours,
        }

    def __repr__(self):
        return f'<ConfigDept {self.department} - {self.line}>'


class ConfigST(db.Model):
    """Standard hours configuration for products"""
    __tablename__ = 'configs_st'

    id = db.Column(db.Integer, primary_key=True)
    product_series = db.Column(db.String(100), nullable=False)
    model = db.Column(db.String(100), nullable=False)
    direct_st = db.Column(db.Float, nullable=False, default=0.0)
    indirect_st = db.Column(db.Float, nullable=False, default=0.0)
    applicable_line = db.Column(db.String(100), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)

    def to_dict(self):
        """Convert to dictionary"""
        return {
            'id': self.id,
            'product_series': self.product_series,
            'model': self.model,
            'direct_st': self.direct_st,
            'indirect_st': self.indirect_st,
            'applicable_line': self.applicable_line,
            'total_st': self.direct_st + self.indirect_st,
        }

    def __repr__(self):
        return f'<ConfigST {self.product_series} - {self.model}>'


class ConfigAbnormal(db.Model):
    """Abnormal type configuration"""
    __tablename__ = 'configs_abnormal'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), unique=True, nullable=False)
    description = db.Column(db.String(255), nullable=True)
    is_active = db.Column(db.Boolean, default=True)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    def to_dict(self):
        """Convert to dictionary"""
        return {
            'id': self.id,
            'name': self.name,
            'description': self.description,
            'is_active': self.is_active,
        }

    def __repr__(self):
        return f'<ConfigAbnormal {self.name}>'


class DailyReport(db.Model):
    """Daily report header"""
    __tablename__ = 'daily_reports'

    id = db.Column(db.Integer, primary_key=True)
    report_date = db.Column(db.Date, nullable=False)
    department = db.Column(db.String(100), nullable=False)
    line = db.Column(db.String(100), nullable=False)
    expected_direct_manpower = db.Column(db.Integer, nullable=False)
    expected_indirect_manpower = db.Column(db.Integer, nullable=False)
    actual_direct_attendance = db.Column(db.Integer, nullable=False)
    actual_indirect_attendance = db.Column(db.Integer, nullable=False)
    direct_overtime_hours = db.Column(db.Float, nullable=False, default=0.0)
    indirect_overtime_hours = db.Column(db.Float, nullable=False, default=0.0)
    direct_adjustment_hours = db.Column(db.Float, nullable=False, default=0.0)
    indirect_adjustment_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Direct input hours
    direct_input_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Indirect input hours
    indirect_input_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Total input hours
    total_input_hours = db.Column(db.Float, nullable=False)
    # Direct output hours
    direct_output_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Indirect output hours
    indirect_output_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Total output hours
    total_output_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Direct abnormal hours
    direct_abnormal_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Indirect abnormal hours
    indirect_abnormal_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Total abnormal hours
    total_abnormal_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Overall efficiency (%)
    overall_efficiency = db.Column(db.Float, nullable=True)
    # Direct efficiency (%)
    direct_efficiency = db.Column(db.Float, nullable=True)
    # Indirect efficiency (%)
    indirect_efficiency = db.Column(db.Float, nullable=True)
    # Abnormal hours rate (%)
    abnormal_hours_rate = db.Column(db.Float, nullable=True)
    created_by = db.Column(db.String(80), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)

    # Relationships
    output_details = db.relationship('DailyOutputDetail', backref='report', lazy='dynamic',
                                    cascade='all, delete-orphan')
    abnormal_details = db.relationship('DailyAbnormalDetail', backref='report', lazy='dynamic',
                                      cascade='all, delete-orphan')

    # Unique constraint on report_date, department, and line combination
    __table_args__ = (
        db.UniqueConstraint('report_date', 'department', 'line', name='uq_report_date_dept_line'),
        db.Index('idx_daily_reports_date', 'report_date'),
        db.Index('idx_daily_reports_dept_line', 'department', 'line'),
    )

    def to_dict(self):
        """Convert to dictionary"""
        return {
            'id': self.id,
            'report_date': self.report_date.strftime('%Y-%m-%d') if self.report_date else None,
            'department': self.department,
            'line': self.line,
            'expected_direct_manpower': self.expected_direct_manpower,
            'expected_indirect_manpower': self.expected_indirect_manpower,
            'actual_direct_attendance': self.actual_direct_attendance,
            'actual_indirect_attendance': self.actual_indirect_attendance,
            'direct_overtime_hours': self.direct_overtime_hours,
            'indirect_overtime_hours': self.indirect_overtime_hours,
            'direct_adjustment_hours': self.direct_adjustment_hours,
            'indirect_adjustment_hours': self.indirect_adjustment_hours,
            'direct_input_hours': self.direct_input_hours,
            'indirect_input_hours': self.indirect_input_hours,
            'total_input_hours': self.total_input_hours,
            'direct_output_hours': self.direct_output_hours,
            'indirect_output_hours': self.indirect_output_hours,
            'total_output_hours': self.total_output_hours,
            'direct_abnormal_hours': self.direct_abnormal_hours,
            'indirect_abnormal_hours': self.indirect_abnormal_hours,
            'total_abnormal_hours': self.total_abnormal_hours,
            'overall_efficiency': self.overall_efficiency,
            'direct_efficiency': self.direct_efficiency,
            'indirect_efficiency': self.indirect_efficiency,
            'abnormal_hours_rate': self.abnormal_hours_rate,
            'created_by': self.created_by,
            'created_at': self.created_at.strftime('%Y-%m-%d %H:%M:%S') if self.created_at else None,
        }

    def __repr__(self):
        return f'<DailyReport {self.report_date} - {self.department} - {self.line}>'


class DailyOutputDetail(db.Model):
    """Daily report output details"""
    __tablename__ = 'daily_output_details'

    id = db.Column(db.Integer, primary_key=True)
    report_id = db.Column(db.Integer, db.ForeignKey('daily_reports.id', ondelete='CASCADE'),
                         nullable=False)
    product_series = db.Column(db.String(100), nullable=False)
    model = db.Column(db.String(100), nullable=False)
    output_quantity = db.Column(db.Integer, nullable=False, default=0)
    direct_st = db.Column(db.Float, nullable=False)
    indirect_st = db.Column(db.Float, nullable=False)
    # Direct output hours = quantity × direct_st
    direct_output_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Indirect output hours = quantity × indirect_st
    indirect_output_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Total output hours = direct_output_hours + indirect_output_hours
    output_hours = db.Column(db.Float, nullable=False, default=0.0)

    __table_args__ = (
        db.Index('idx_daily_output_report', 'report_id'),
    )

    def to_dict(self):
        """Convert to dictionary"""
        return {
            'id': self.id,
            'report_id': self.report_id,
            'product_series': self.product_series,
            'model': self.model,
            'output_quantity': self.output_quantity,
            'direct_st': self.direct_st,
            'indirect_st': self.indirect_st,
            'direct_output_hours': self.direct_output_hours,
            'indirect_output_hours': self.indirect_output_hours,
            'output_hours': self.output_hours,
        }

    def __repr__(self):
        return f'<DailyOutputDetail {self.report_id} - {self.model}>'


class DailyAbnormalDetail(db.Model):
    """Daily report abnormal details"""
    __tablename__ = 'daily_abnormal_details'

    id = db.Column(db.Integer, primary_key=True)
    report_id = db.Column(db.Integer, db.ForeignKey('daily_reports.id', ondelete='CASCADE'),
                         nullable=False)
    abnormal_type = db.Column(db.String(100), nullable=False)
    # Description of the abnormal impact
    description = db.Column(db.Text, nullable=True)
    # Direct abnormal hours
    direct_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Indirect abnormal hours
    indirect_hours = db.Column(db.Float, nullable=False, default=0.0)
    # Total lost hours
    lost_hours = db.Column(db.Float, nullable=False, default=0.0)

    __table_args__ = (
        db.Index('idx_daily_abnormal_report', 'report_id'),
    )

    def to_dict(self):
        """Convert to dictionary"""
        return {
            'id': self.id,
            'report_id': self.report_id,
            'abnormal_type': self.abnormal_type,
            'description': self.description,
            'direct_hours': self.direct_hours,
            'indirect_hours': self.indirect_hours,
            'lost_hours': self.lost_hours,
        }

    def __repr__(self):
        return f'<DailyAbnormalDetail {self.report_id} - {self.abnormal_type}>'
