"""
Database initialization script
Creates all tables and adds default data
"""
import sys
import os

# Add parent directory to path to import app modules
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from database.models import db, User, ConfigAbnormal
from config import SQLALCHEMY_DATABASE_URI
from flask import Flask


def init_database():
    """Initialize the database with all tables and default data"""
    app = Flask(__name__)
    app.config['SQLALCHEMY_DATABASE_URI'] = SQLALCHEMY_DATABASE_URI
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False

    db.init_app(app)

    with app.app_context():
        # Drop all existing tables (use with caution!)
        print("Dropping existing tables...")
        db.drop_all()

        # Create all tables
        print("Creating tables...")
        db.create_all()

        # Create default admin user
        print("Creating default admin user...")
        admin = User(
            username='admin',
            role='admin'
        )
        admin.set_password('admin123')
        db.session.add(admin)

        # Create default lead user for testing
        print("Creating default lead user...")
        lead = User(
            username='lead',
            role='lead'
        )
        lead.set_password('lead123')
        db.session.add(lead)

        # Create default abnormal types
        print("Creating default abnormal types...")
        abnormal_types = [
            ConfigAbnormal(name='待料', description='Material shortage'),
            ConfigAbnormal(name='机障', description='Machine breakdown'),
            ConfigAbnormal(name='品质', description='Quality issues'),
            ConfigAbnormal(name='换模', description='Model changeover'),
            ConfigAbnormal(name='培训', description='Training'),
            ConfigAbnormal(name='会议', description='Meeting'),
            ConfigAbnormal(name='其他', description='Other'),
        ]
        for abnormal_type in abnormal_types:
            db.session.add(abnormal_type)

        # Commit all changes
        db.session.commit()

        print("\nDatabase initialized successfully!")
        print("\nDefault users created:")
        print("  Admin - Username: admin, Password: admin123")
        print("  Lead  - Username: lead, Password: lead123")
        print("\nDefault abnormal types created:")
        for at in abnormal_types:
            print(f"  - {at.name} ({at.description})")

        # Print database location
        print(f"\nDatabase location: {SQLALCHEMY_DATABASE_URI.replace('sqlite:///', '')}")


if __name__ == '__main__':
    init_database()
