"""
Main Flask application for the Factory Efficiency Management System
"""
import os
from datetime import timedelta
from flask import Flask, render_template, redirect, url_for, session, flash
from flask_login import LoginManager, login_required, current_user
from waitress import serve

from config import (
    SECRET_KEY, SQLALCHEMY_DATABASE_URI, SQLALCHEMY_TRACK_MODIFICATIONS,
    HOST, PORT, DEBUG, PERMANENT_SESSION_LIFETIME
)
from database.models import db, User

# Create Flask app
app = Flask(__name__)

# Configuration
app.config['SECRET_KEY'] = SECRET_KEY
app.config['SQLALCHEMY_DATABASE_URI'] = SQLALCHEMY_DATABASE_URI
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = SQLALCHEMY_TRACK_MODIFICATIONS
app.config['PERMANENT_SESSION_LIFETIME'] = timedelta(seconds=PERMANENT_SESSION_LIFETIME)

# Initialize database
db.init_app(app)

# Initialize login manager
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'auth.login'
login_manager.login_message = 'Please login to access this page.'
login_manager.login_message_category = 'warning'


@login_manager.user_loader
def load_user(user_id):
    """Load user by ID for Flask-Login"""
    return db.session.get(User, int(user_id))


# Register blueprints
from routes import auth_bp, config_bp, daily_report_bp, dashboard_bp, api_bp

app.register_blueprint(auth_bp)
app.register_blueprint(config_bp, url_prefix='/config')
app.register_blueprint(daily_report_bp, url_prefix='/report')
app.register_blueprint(dashboard_bp, url_prefix='/dashboard')
app.register_blueprint(api_bp, url_prefix='/api')


# Context processor to make user info available in all templates
@app.context_processor
def inject_user():
    """Inject current user into all templates"""
    return dict(current_user=current_user)


@app.route('/')
@login_required
def index():
    """Main dashboard/index page"""
    if current_user.is_admin():
        return render_template('index.html', is_admin=True)
    else:
        return render_template('index.html', is_admin=False)


@app.errorhandler(404)
def not_found(error):
    """Handle 404 errors"""
    return render_template('error.html', error='Page not found'), 404


@app.errorhandler(500)
def internal_error(error):
    """Handle 500 errors"""
    db.session.rollback()
    return render_template('error.html', error='Internal server error'), 500


def create_tables():
    """Create database tables if they don't exist"""
    with app.app_context():
        db.create_all()


def run_production():
    """Run the app with Waitress production server"""
    serve(app, host=HOST, port=PORT)


def run_development():
    """Run the app in development mode"""
    app.run(host=HOST, port=PORT, debug=DEBUG)


if __name__ == '__main__':
    # Create tables if they don't exist
    create_tables()
    print("=" * 50)
    print("Factory Efficiency Management System")
    print("=" * 50)
    print(f"Server running at http://{HOST}:{PORT}")
    print(f"Database: {SQLALCHEMY_DATABASE_URI.replace('sqlite:///', '')}")
    print("=" * 50)

    # Run with Waitress for production
    run_production()
