#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Andon 数据库模型
"""
from flask_sqlalchemy import SQLAlchemy
from datetime import datetime
from werkzeug.security import generate_password_hash, check_password_hash

db = SQLAlchemy()


class Station(db.Model):
    """工位配置表"""
    __tablename__ = 'stations'

    id = db.Column(db.Integer, primary_key=True)
    station_id = db.Column(db.String(20), unique=True, nullable=False, index=True)  # 登录ID
    station_name = db.Column(db.String(50), nullable=False)  # 显示名称
    password_hash = db.Column(db.String(255), nullable=False)
    is_active = db.Column(db.Boolean, default=True)
    created_at = db.Column(db.DateTime, default=datetime.now)

    def set_password(self, password):
        """设置密码（加密存储）"""
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        """验证密码"""
        return check_password_hash(self.password_hash, password)

    def to_dict(self):
        """转换为字典"""
        return {
            'id': self.id,
            'station_id': self.station_id,
            'station_name': self.station_name,
            'is_active': self.is_active
        }


class IssueCategory(db.Model):
    """异常类型配置表"""
    __tablename__ = 'issue_categories'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(20), unique=True, nullable=False)  # 类型名称
    color = db.Column(db.String(20), default='primary')  # Bootstrap 颜色
    display_order = db.Column(db.Integer, default=0)  # 显示顺序
    created_at = db.Column(db.DateTime, default=datetime.now)

    def to_dict(self):
        """转换为字典"""
        return {
            'id': self.id,
            'name': self.name,
            'color': self.color,
            'display_order': self.display_order
        }


class AndonIssue(db.Model):
    """异常记录表"""
    __tablename__ = 'andon_issues'

    id = db.Column(db.Integer, primary_key=True)
    station_id = db.Column(db.Integer, db.ForeignKey('stations.id'), nullable=False)
    station_name = db.Column(db.String(50), nullable=False)
    category_id = db.Column(db.Integer, db.ForeignKey('issue_categories.id'), nullable=False)
    category_name = db.Column(db.String(20), nullable=False)
    category_color = db.Column(db.String(20), default='primary')  # 冗余存储，方便查询
    status = db.Column(db.String(20), default='Open', index=True)  # Open, Closed
    start_time = db.Column(db.DateTime, default=datetime.now, index=True)
    end_time = db.Column(db.DateTime, nullable=True)
    solution_note = db.Column(db.Text, nullable=True)
    response_duration = db.Column(db.Integer, nullable=True)  # 秒

    station = db.relationship('Station', backref='issues')
    category = db.relationship('IssueCategory', backref='issues')

    def resolve(self, solution_note=''):
        """解决异常"""
        from datetime import datetime
        self.status = 'Closed'
        self.end_time = datetime.now()
        self.solution_note = solution_note
        if self.start_time:
            delta = self.end_time - self.start_time
            self.response_duration = int(delta.total_seconds())

    def to_dict(self):
        """转换为字典"""
        return {
            'id': self.id,
            'station_id': self.station_id,
            'station_name': self.station_name,
            'category_id': self.category_id,
            'category_name': self.category_name,
            'category_color': self.category_color,
            'status': self.status,
            'start_time': self.start_time.isoformat() if self.start_time else None,
            'end_time': self.end_time.isoformat() if self.end_time else None,
            'solution_note': self.solution_note,
            'response_duration': self.response_duration
        }

    def get_duration_minutes(self):
        """获取持续分钟数"""
        if self.status == 'Closed' and self.end_time:
            delta = self.end_time - self.start_time
        else:
            delta = datetime.now() - self.start_time
        return int(delta.total_seconds() / 60)
