#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Andon 安灯系统主应用
"""
import os
from datetime import datetime
from flask import Flask, render_template, request, jsonify, session, redirect, url_for
from flask_sqlalchemy import SQLAlchemy
from models import db, Station, IssueCategory, AndonIssue

app = Flask(__name__)
app.config['SECRET_KEY'] = 'andon-secret-key-change-in-production'
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///andon.db'
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False

db.init_app(app)


# ==================== 初始化数据 ====================

def init_db():
    """初始化数据库，创建默认数据"""
    with app.app_context():
        # 创建所有表
        db.create_all()

        # 检查是否已有工位数据
        if Station.query.count() == 0:
            # 创建默认工位账号
            admin = Station(station_id='admin', station_name='管理员工位')
            admin.set_password('admin123')
            db.session.add(admin)

            # 创建示例工位
            station1 = Station(station_id='station01', station_name='工位01')
            station1.set_password('123456')
            db.session.add(station1)

            station2 = Station(station_id='station02', station_name='工位02')
            station2.set_password('123456')
            db.session.add(station2)

        # 检查是否已有异常类型数据
        if IssueCategory.query.count() == 0:
            # 创建默认异常类型
            categories = [
                IssueCategory(name='品质', color='danger', display_order=1),   # 红色
                IssueCategory(name='设备', color='primary', display_order=2),  # 蓝色
                IssueCategory(name='材料', color='warning', display_order=3),  # 黄色
                IssueCategory(name='人员', color='success', display_order=4),  # 绿色
                IssueCategory(name='工艺', color='info', display_order=5),     # 青色
            ]
            for cat in categories:
                db.session.add(cat)

        db.session.commit()
        print("数据库初始化完成！")


# ==================== 辅助函数 ====================

def get_logged_in_station():
    """获取当前登录的工位"""
    if 'station_id' in session:
        return Station.query.get(session['station_id'])
    return None


def is_config_logged_in():
    """检查配置页面是否已登录"""
    return session.get('config_logged_in', False)


# ==================== 页面路由 ====================

@app.route('/')
def index():
    """工位登录页/操作页"""
    station = get_logged_in_station()
    if station:
        # 已登录，显示操作界面
        categories = IssueCategory.query.order_by(IssueCategory.display_order).all()
        my_issues = AndonIssue.query.filter_by(
            station_id=station.id,
            status='Open'
        ).order_by(AndonIssue.start_time.desc()).all()
        return render_template('index.html',
                             station=station,
                             categories=categories,
                             my_issues=my_issues)
    else:
        # 未登录，显示登录表单
        return render_template('index.html')


@app.route('/login', methods=['POST'])
def login():
    """工位登录"""
    station_id = request.form.get('station_id')
    password = request.form.get('password')

    station = Station.query.filter_by(station_id=station_id).first()

    if station and station.check_password(password):
        if not station.is_active:
            return jsonify({'success': False, 'message': '该工位已被禁用'})
        session['station_id'] = station.id
        session['station_name'] = station.station_name
        return jsonify({'success': True, 'message': '登录成功'})
    else:
        return jsonify({'success': False, 'message': '工位ID或密码错误'})


@app.route('/logout', methods=['POST'])
def logout():
    """工位登出"""
    session.clear()
    return jsonify({'success': True, 'message': '已登出'})


@app.route('/dashboard')
def dashboard():
    """车间看板页"""
    return render_template('dashboard.html')


@app.route('/history')
def history():
    """历史记录页"""
    stations = Station.query.filter_by(is_active=True).order_by(Station.station_name).all()
    categories = IssueCategory.query.order_by(IssueCategory.display_order).all()
    return render_template('history.html', stations=stations, categories=categories)


@app.route('/config')
def config():
    """系统配置页"""
    if not is_config_logged_in():
        return render_template('config.html', show_login=True)

    stations = Station.query.order_by(Station.station_id).all()
    categories = IssueCategory.query.order_by(IssueCategory.display_order).all()
    return render_template('config.html', stations=stations, categories=categories)


@app.route('/config/login', methods=['POST'])
def config_login():
    """配置页登录"""
    password = request.form.get('password')
    # 使用默认管理员密码验证
    admin = Station.query.filter_by(station_id='admin').first()
    if admin and admin.check_password(password):
        session['config_logged_in'] = True
        return jsonify({'success': True, 'message': '登录成功'})
    return jsonify({'success': False, 'message': '密码错误'})


@app.route('/config/logout', methods=['POST'])
def config_logout():
    """配置页登出"""
    session.pop('config_logged_in', None)
    return jsonify({'success': True})


# ==================== API 路由 ====================

@app.route('/api/trigger', methods=['POST'])
def api_trigger():
    """创建新异常"""
    station = get_logged_in_station()
    if not station:
        return jsonify({'success': False, 'message': '请先登录'})

    data = request.get_json()
    category_id = data.get('category_id')

    category = IssueCategory.query.get(category_id)
    if not category:
        return jsonify({'success': False, 'message': '异常类型不存在'})

    issue = AndonIssue(
        station_id=station.id,
        station_name=station.station_name,
        category_id=category.id,
        category_name=category.name,
        category_color=category.color
    )
    db.session.add(issue)
    db.session.commit()

    return jsonify({
        'success': True,
        'message': '呼叫成功！',
        'issue': issue.to_dict()
    })


@app.route('/api/resolve/<int:issue_id>', methods=['POST'])
def api_resolve(issue_id):
    """解决异常"""
    station = get_logged_in_station()
    if not station:
        return jsonify({'success': False, 'message': '请先登录'})

    issue = AndonIssue.query.get(issue_id)
    if not issue:
        return jsonify({'success': False, 'message': '异常记录不存在'})

    # 只能解决自己的异常
    if issue.station_id != station.id:
        return jsonify({'success': False, 'message': '只能解决自己工位的异常'})

    if issue.status != 'Open':
        return jsonify({'success': False, 'message': '该异常已解决'})

    data = request.get_json()
    solution_note = data.get('solution_note', '')

    issue.resolve(solution_note)
    db.session.commit()

    return jsonify({
        'success': True,
        'message': '异常已解决',
        'issue': issue.to_dict()
    })


@app.route('/api/active_issues')
def api_active_issues():
    """获取所有 Open 状态的异常"""
    issues = AndonIssue.query.filter_by(status='Open').order_by(
        AndonIssue.start_time.desc()
    ).all()
    return jsonify({
        'success': True,
        'issues': [issue.to_dict() for issue in issues]
    })


@app.route('/api/my_issues')
def api_my_issues():
    """获取当前工位的 Open 异常"""
    station = get_logged_in_station()
    if not station:
        return jsonify({'success': False, 'message': '请先登录'})

    issues = AndonIssue.query.filter_by(
        station_id=station.id,
        status='Open'
    ).order_by(AndonIssue.start_time.desc()).all()

    return jsonify({
        'success': True,
        'issues': [issue.to_dict() for issue in issues]
    })


@app.route('/api/history')
def api_history():
    """获取历史数据（包含未解决的记录）"""
    station_id = request.args.get('station_id', type=int)
    category_id = request.args.get('category_id', type=int)
    status = request.args.get('status', type=str)  # 可选的状态筛选

    query = AndonIssue.query

    if station_id:
        query = query.filter_by(station_id=station_id)
    if category_id:
        query = query.filter_by(category_id=category_id)
    if status:  # 如果指定了状态筛选
        query = query.filter_by(status=status)

    issues = query.order_by(AndonIssue.start_time.desc()).limit(100).all()

    return jsonify({
        'success': True,
        'issues': [issue.to_dict() for issue in issues]
    })


# ==================== 配置管理 API ====================

@app.route('/api/config/stations', methods=['GET', 'POST'])
def api_config_stations():
    """工位配置管理"""
    if not is_config_logged_in():
        return jsonify({'success': False, 'message': '请先登录'})

    if request.method == 'GET':
        stations = Station.query.order_by(Station.station_id).all()
        return jsonify({
            'success': True,
            'stations': [s.to_dict() for s in stations]
        })

    # POST - 添加/编辑工位
    data = request.get_json()
    station_id = data.get('id')

    if station_id:
        # 编辑现有工位
        station = Station.query.get(station_id)
        if not station:
            return jsonify({'success': False, 'message': '工位不存在'})
        station.station_id = data.get('station_id')
        station.station_name = data.get('station_name')
        if data.get('password'):
            station.set_password(data.get('password'))
        station.is_active = data.get('is_active', True)
    else:
        # 新增工位
        station = Station(
            station_id=data.get('station_id'),
            station_name=data.get('station_name')
        )
        station.set_password(data.get('password', '123456'))
        station.is_active = data.get('is_active', True)
        db.session.add(station)

    try:
        db.session.commit()
        return jsonify({'success': True, 'message': '保存成功'})
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': f'保存失败: {str(e)}'})


@app.route('/api/config/stations/<int:station_id>', methods=['DELETE'])
def api_config_delete_station(station_id):
    """删除工位"""
    if not is_config_logged_in():
        return jsonify({'success': False, 'message': '请先登录'})

    station = Station.query.get(station_id)
    if not station:
        return jsonify({'success': False, 'message': '工位不存在'})

    if station.station_id == 'admin':
        return jsonify({'success': False, 'message': '不能删除管理员账号'})

    db.session.delete(station)
    db.session.commit()
    return jsonify({'success': True, 'message': '删除成功'})


@app.route('/api/config/categories', methods=['GET', 'POST'])
def api_config_categories():
    """异常类型管理"""
    if not is_config_logged_in():
        return jsonify({'success': False, 'message': '请先登录'})

    if request.method == 'GET':
        categories = IssueCategory.query.order_by(IssueCategory.display_order).all()
        return jsonify({
            'success': True,
            'categories': [c.to_dict() for c in categories]
        })

    # POST - 添加/编辑异常类型
    data = request.get_json()
    category_id = data.get('id')

    if category_id:
        # 编辑
        category = IssueCategory.query.get(category_id)
        if not category:
            return jsonify({'success': False, 'message': '类型不存在'})
        category.name = data.get('name')
        category.color = data.get('color')
        category.display_order = data.get('display_order', 0)
    else:
        # 新增
        category = IssueCategory(
            name=data.get('name'),
            color=data.get('color'),
            display_order=data.get('display_order', 0)
        )
        db.session.add(category)

    try:
        db.session.commit()
        return jsonify({'success': True, 'message': '保存成功'})
    except Exception as e:
        db.session.rollback()
        return jsonify({'success': False, 'message': f'保存失败: {str(e)}'})


@app.route('/api/config/categories/<int:category_id>', methods=['DELETE'])
def api_config_delete_category(category_id):
    """删除异常类型"""
    if not is_config_logged_in():
        return jsonify({'success': False, 'message': '请先登录'})

    category = IssueCategory.query.get(category_id)
    if not category:
        return jsonify({'success': False, 'message': '类型不存在'})

    # 检查是否有使用该类型的异常记录
    if AndonIssue.query.filter_by(category_id=category_id).count() > 0:
        return jsonify({'success': False, 'message': '该类型已被使用，无法删除'})

    db.session.delete(category)
    db.session.commit()
    return jsonify({'success': True, 'message': '删除成功'})


# ==================== 启动应用 ====================

if __name__ == '__main__':
    # 初始化数据库
    init_db()
    # 运行应用
    app.run(host='0.0.0.0', port=5001, debug=True)
